import {
  getOperator,
  initOptions,
  OpType,
  ProcessingMode
} from "./core";
import { Lazy } from "./lazy";
import { assert, cloneDeep } from "./util";
class Aggregator {
  #pipeline;
  #options;
  /**
   * Creates an instance of the Aggregator class.
   *
   * @param pipeline - An array of objects representing the aggregation pipeline stages.
   * @param options - Optional configuration settings for the aggregator.
   */
  constructor(pipeline, options) {
    this.#pipeline = pipeline;
    this.#options = initOptions(options);
  }
  /**
   * Processes a collection through an aggregation pipeline and returns an iterator
   * for the transformed results.
   *
   * @param collection - The input collection to process. This can be any source
   *   that implements the `Source` interface.
   * @param options - Optional configuration for processing. If not provided, the
   *   default options of the aggregator instance will be used.
   * @returns An iterator that yields the results of the aggregation pipeline.
   *
   * @throws Will throw an error if:
   * - A pipeline stage contains more than one operator.
   * - The `$documents` operator is not the first stage in the pipeline.
   * - An unregistered pipeline operator is encountered.
   */
  stream(collection, options) {
    let iter = Lazy(collection);
    const opts = options ?? this.#options;
    const mode = opts.processingMode;
    if (mode & ProcessingMode.CLONE_INPUT) iter.map(cloneDeep);
    iter = this.#pipeline.map((stage, i) => {
      const keys = Object.keys(stage);
      assert(
        keys.length === 1,
        `aggregation stage must have single operator, got ${keys.toString()}.`
      );
      const name = keys[0];
      assert(
        name !== "$documents" || i == 0,
        "$documents must be first stage in pipeline."
      );
      const op = getOperator(OpType.PIPELINE, name, opts);
      assert(!!op, `unregistered pipeline operator ${name}.`);
      return [op, stage[name]];
    }).reduce((acc, [op, expr]) => op(acc, expr, opts), iter);
    if (mode & ProcessingMode.CLONE_OUTPUT) iter.map(cloneDeep);
    return iter;
  }
  /**
   * Executes the aggregation pipeline on the provided collection and returns the resulting array.
   *
   * @template T - The type of the objects in the resulting array.
   * @param collection - The input data source to run the aggregation on.
   * @param options - Optional settings to customize the aggregation behavior.
   * @returns An array of objects of type `T` resulting from the aggregation.
   */
  run(collection, options) {
    return this.stream(collection, options).value();
  }
}
export {
  Aggregator
};
