import { CollationSpec, Options } from "./core";
import { Iterator, Source } from "./lazy";
import { Any, AnyObject, Callback, Predicate } from "./types";
/**
 * The `Cursor` class provides a mechanism for iterating over a collection of data
 * with support for filtering, projection, sorting, skipping, and limiting results.
 * It is designed to be chainable and supports lazy evaluation for efficient data processing.
 *
 * @template T - The type of the elements in the cursor.
 */
export declare class Cursor<T> {
    #private;
    /**
     * Creates an instance of the Cursor class.
     *
     * @param source - The source of data to be iterated over.
     * @param predicate - A function or condition to filter the data.
     * @param projection - An object specifying the fields to include or exclude in the result.
     * @param options - Optional settings to customize the behavior of the cursor.
     */
    constructor(source: Source, predicate: Predicate<Any>, projection: AnyObject, options?: Options);
    /** Returns the iterator from running the query */
    private fetch;
    /** Returns an iterator with the buffered data included */
    private fetchAll;
    /**
     * Return remaining objects in the cursor as an array. This method exhausts the cursor
     * @returns {Array}
     */
    all(): T[];
    /**
     * Returns the number of objects return in the cursor. This method exhausts the cursor
     * @returns {Number}
     */
    count(): number;
    /**
     * Returns a cursor that begins returning results only after passing or skipping a number of documents.
     * @param {Number} n the number of results to skip.
     * @return {Cursor} Returns the cursor, so you can chain this call.
     */
    skip(n: number): Cursor<T>;
    /**
     * Limits the number of items returned by the cursor.
     *
     * @param n - The maximum number of items to return.
     * @returns The current cursor instance for chaining.
     */
    limit(n: number): Cursor<T>;
    /**
     * Returns results ordered according to a sort specification.
     * @param {AnyObject} modifier an object of key and values specifying the sort order. 1 for ascending and -1 for descending
     * @return {Cursor} Returns the cursor, so you can chain this call.
     */
    sort(modifier: AnyObject): Cursor<T>;
    /**
     * Applies a sort operation to the cursor using the specified sort modifier.
     *
     * @param modifier - An object specifying the sort order. The keys represent
     * the field names, and the values indicate the sort direction (e.g., 1 for
     * ascending and -1 for descending).
     * @returns The current cursor instance for method chaining.
     */
    collation(spec: CollationSpec): Cursor<T>;
    /**
     * Retrieves the next item in the cursor.
     *
     * If there are items in the internal buffer, the next item is returned from the buffer.
     * Otherwise, it fetches the next item from the underlying data source.
     *
     * @returns The next item of type `T` if available, or `undefined` if there are no more items.
     */
    next(): T;
    /**
     * Determines if there are more elements available in the cursor.
     *
     * This method checks if there are any elements left in the internal buffer.
     * If the buffer is empty, it attempts to fetch the next element from the source.
     * If a new element is found, it is added to the buffer and the method returns `true`.
     * Otherwise, it returns `false` indicating no more elements are available.
     *
     * @returns {boolean} `true` if there are more elements to iterate over, otherwise `false`.
     */
    hasNext(): boolean;
    /**
     * Transforms the elements of the cursor using the provided callback function.
     *
     * @template R - The type of the elements in the resulting array.
     * @template T - The type of the elements in the cursor.
     * @param fn - A callback function that is invoked for each element in the cursor.
     *             It receives the current element, its index, and the entire array as arguments.
     * @returns An array of transformed elements of type `R`.
     */
    map<R>(fn: Callback<R, T>): R[];
    /**
     * Iterates over all elements in the cursor and executes the provided callback function for each element.
     *
     * @param fn - A callback function to execute for each element. The function receives the following arguments:
     *   - `t`: The current element being processed in the cursor.
     *   - `i`: The index of the current element in the array.
     *   - `a`: The array of all elements in the cursor.
     */
    forEach(fn: Callback<void, T>): void;
    /**
     * Returns an iterator for the cursor, allowing it to be used in `for...of` loops.
     * The iterator fetches all the results from the cursor.
     *
     * @returns {Iterator} An iterator over the fetched results.
     */
    [Symbol.iterator](): Iterator;
}
