import { Any, Callback, Predicate } from "./types";
/**
 * A value produced by a generator
 */
export interface IteratorResult<T = Any> {
    readonly value?: T;
    readonly done: boolean;
}
/**
 * Represents a stream interface that provides a method to retrieve the next item in a sequence.
 */
interface Stream {
    next: () => IteratorResult;
}
export type Source = Stream | Callback<IteratorResult> | Iterable<Any>;
/**
 * Returns an iterator
 * @param {*} source An iterable source (Array, Function, Generator, or Iterator)
 */
export declare function Lazy(source: Source): Iterator;
/**
 * Concatenate multiple iterators and return a new iterator.
 *
 * @param iterators The iterators to concatenate
 * @returns {Iterator} A new iterator
 */
export declare function concat(...iterators: Iterator[]): Iterator;
/**
 * A lazy collection iterator yields a single value at a time upon request.
 */
export declare class Iterator {
    #private;
    constructor(source: Source);
    /**
     * Add an iteratee to this lazy sequence
     */
    private push;
    next<T = Any>(): IteratorResult<T>;
    /**
     * Transform each item in the sequence to a new value
     * @param {Function} f
     */
    map<T = Any>(f: Callback<T>): Iterator;
    /**
     * Select only items matching the given predicate
     * @param {Function} f
     */
    filter<T = Any>(f: Predicate<T>): Iterator;
    /**
     * Take given numbe for values from sequence
     * @param {Number} n A number greater than 0
     */
    take(n: number): Iterator;
    /**
     * Drop a number of values from the sequence
     * @param {Number} n Number of items to drop greater than 0
     */
    drop(n: number): Iterator;
    /**
     * Returns a new lazy object with results of the transformation
     * The entire sequence is realized.
     *
     * @param {Callback<Source, Any[]>} fn Tranform function of type (Array) => (Any)
     */
    transform(fn: Callback<Source, Any[]>): Iterator;
    /**
     * Returns the fully realized values of the iterators.
     * The return value will be an array unless `lazy.first()` was used.
     * The realized values are cached for subsequent calls.
     */
    value<T>(): T[];
    /**
     * Execute the funcion for each value. Will stop when an execution returns false.
     * @param {Function} f
     * @returns {Boolean} false iff `f` return false for AnyVal execution, otherwise true
     */
    each<T = Any>(f: Callback<T>): boolean;
    /**
     * Returns the reduction of sequence according the reducing function
     *
     * @param {*} f a reducing function
     * @param {*} initialValue
     */
    reduce<T = Any>(f: Callback<T>, initialValue?: T): T;
    /**
     * Returns the number of matched items in the sequence
     */
    size(): number;
    [Symbol.iterator](): Iterator;
}
export {};
