/**
 * Predicates used for Query and Expression operators.
 */
import { Options } from "../core";
import { Any, AnyObject, BsonType, JsType, Predicate } from "../types";
type PredicateOptions = Options & {
    depth: number;
};
type ConversionType = number | Exclude<JsType, "function"> | BsonType;
export declare function processQuery(selector: string, value: Any, options: Options, predicate: Predicate<Any>): (_: AnyObject) => boolean;
export declare function processExpression(obj: AnyObject, expr: Any, options: Options, predicate: Predicate<Any>): boolean;
/**
 * Checks that two values are equal.
 *
 * @param a         The lhs operand as resolved from the object by the given selector
 * @param b         The rhs operand provided by the user
 * @returns {*}
 */
export declare function $eq(a: Any, b: Any, options?: PredicateOptions): boolean;
/**
 * Matches all values that are not equal to the value specified in the query.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $ne(a: Any, b: Any, options?: PredicateOptions): boolean;
/**
 * Matches any of the values that exist in an array specified in the query.
 *
 * @param a
 * @param b
 * @returns {*}
 */
export declare function $in(a: Any[], b: Any[], options?: PredicateOptions): boolean;
/**
 * Matches values that do not exist in an array specified to the query.
 *
 * @param a
 * @param b
 * @returns {*|boolean}
 */
export declare function $nin(a: Any[], b: Any[], options?: PredicateOptions): boolean;
/**
 * Matches values that are less than the value specified in the query.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $lt(a: Any, b: Any, _options?: PredicateOptions): boolean;
/**
 * Matches values that are less than or equal to the value specified in the query.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $lte(a: Any, b: Any, _options?: PredicateOptions): boolean;
/**
 * Matches values that are greater than the value specified in the query.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $gt(a: Any, b: Any, _options?: PredicateOptions): boolean;
/**
 * Matches values that are greater than or equal to the value specified in the query.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $gte(a: Any, b: Any, _options?: PredicateOptions): boolean;
/**
 * Performs a modulo operation on the value of a field and selects documents with a specified result.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $mod(a: Any, b: number[], _options?: PredicateOptions): boolean;
/**
 * Selects documents where values match a specified regular expression.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $regex(a: Any, b: RegExp, options?: PredicateOptions): boolean;
/**
 * Matches arrays that contain all elements specified in the query.
 *
 * @param values
 * @param queries
 * @returns boolean
 */
export declare function $all(values: Any[], queries: AnyObject[], options?: PredicateOptions): boolean;
/**
 * Selects documents if the array field is a specified size.
 *
 * @param a
 * @param b
 * @returns {*|boolean}
 */
export declare function $size(a: Any[], b: number, _options?: PredicateOptions): boolean;
/**
 * Selects documents if element in the array field matches all the specified $elemMatch condition.
 *
 * @param a {Any[]} element to match against
 * @param b {AnyObject} subquery
 */
export declare function $elemMatch(a: Any[], b: AnyObject, options?: PredicateOptions): boolean;
/**
 * Selects documents if a field is of the specified type.
 *
 * @param a
 * @param b
 * @returns {boolean}
 */
export declare function $type(a: Any, b: ConversionType | ConversionType[], options?: PredicateOptions): boolean;
export {};
