import { Options } from "./core";
import { Cursor } from "./cursor";
import { Source } from "./lazy";
import { AnyObject } from "./types";
/**
 * Represents a query object used to filter and match documents based on specified criteria.
 *
 * The `Query` class provides methods to compile query conditions, test objects against
 * the query criteria, and retrieve matching documents from a collection.
 *
 * @example
 * ```typescript
 * const query = new Query({ age: { $gt: 18 } });
 * const result = query.test({ name: "John", age: 25 }); // true
 * ```
 *
 * @template T - The type of objects being queried.
 */
export declare class Query {
    #private;
    /**
     * Creates an instance of the query with the specified condition and options.
     *
     * @param condition - The query condition object used to define the criteria for matching documents.
     * @param options - Optional configuration settings to customize the query behavior.
     */
    constructor(condition: AnyObject, options?: Partial<Options>);
    private compile;
    private processOperator;
    /**
     * Tests whether the given object satisfies all compiled predicates.
     *
     * @template T - The type of the object to test.
     * @param obj - The object to be tested against the compiled predicates.
     * @returns `true` if the object satisfies all predicates, otherwise `false`.
     */
    test<T>(obj: T): boolean;
    /**
     * Returns a cursor for iterating over the items in the given collection that match the query criteria.
     *
     * @typeParam T - The type of the items in the resulting cursor.
     * @param collection - The source collection to search through.
     * @param projection - An optional object specifying fields to include or exclude
     *                      in the returned items.
     * @returns A `Cursor` instance for iterating over the matching items.
     */
    find<T>(collection: Source, projection?: AnyObject): Cursor<T>;
}
